<?php
// apply_for_job.php

// Include the database connection file
require_once 'db_connect.php'; // Adjust path if db_connect.php is elsewhere

// Set the response header to JSON
header('Content-Type: application/json');

// --- Define Error Response Function ---
// Helper function to send consistent error JSON (using the specified format if possible)
function send_json_error($message, $http_code = 400) {
    http_response_code($http_code); // Set appropriate HTTP status
    // Use the target JSON structure even for errors, but with an error message
    echo json_encode([
        "data" => [
            // Use a generic key like 'message' or 'error' inside body_text for simplicity
            "body_text" => "Error: " . $message
        ]
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    // Ensure connection is closed if it exists before exiting
    if (isset($GLOBALS['conn']) && $GLOBALS['conn'] instanceof mysqli) {
       mysqli_close($GLOBALS['conn']);
    }
    exit();
}

// --- Get and Validate Input Parameters ---
if (!isset($_GET['job_id']) || !isset($_GET['phone'])) {
    send_json_error("Both job_id and phone parameters are required.", 400);
}

$job_id = $_GET['job_id'];
$phone_raw = trim($_GET['phone']);

// Validate job_id is numeric and phone is not empty
if (!is_numeric($job_id) || empty($phone_raw)) {
    send_json_error("Invalid input: job_id must be numeric and phone cannot be empty.", 400);
}
$job_id = (int)$job_id; // Cast to integer


// --- Normalize Phone Number ---
$phone_normalized = preg_replace('/[^\d+]/', '', $phone_raw);
if (substr($phone_normalized, 0, 1) !== '+') {
    $phone_normalized = '+' . $phone_normalized;
}

// --- 1. Find User ID based on Phone Number ---
$user_id = null;
$sql_user = "SELECT user_id FROM users WHERE phone_number = ? AND is_verified = TRUE LIMIT 1"; // Ensure user is verified
$stmt_user = mysqli_prepare($conn, $sql_user);

if ($stmt_user) {
    mysqli_stmt_bind_param($stmt_user, "s", $phone_normalized);
    if (mysqli_stmt_execute($stmt_user)) {
        $result_user = mysqli_stmt_get_result($stmt_user);
        if ($row_user = mysqli_fetch_assoc($result_user)) {
            $user_id = $row_user['user_id'];
        } else {
            send_json_error("User not found or not verified with this phone number.", 404);
        }
    } else {
        // Log error
        // error_log("Apply Job - User Query Execute Failed: " . mysqli_stmt_error($stmt_user));
        send_json_error("Database error checking user.", 500);
    }
    mysqli_stmt_close($stmt_user);
} else {
    // Log error
    // error_log("Apply Job - User Query Prepare Failed: " . mysqli_error($conn));
    send_json_error("Database error preparing user check.", 500);
}

// --- 2. Find Job Title and check if active ---
$job_title = null;
$sql_job = "SELECT title FROM jobs WHERE job_id = ? AND is_active = TRUE LIMIT 1";
$stmt_job = mysqli_prepare($conn, $sql_job);

if ($stmt_job) {
    mysqli_stmt_bind_param($stmt_job, "i", $job_id);
    if (mysqli_stmt_execute($stmt_job)) {
        $result_job = mysqli_stmt_get_result($stmt_job);
        if ($row_job = mysqli_fetch_assoc($result_job)) {
            $job_title = $row_job['title'];
        } else {
            send_json_error("Job not found or is no longer active.", 404);
        }
    } else {
        // Log error
        // error_log("Apply Job - Job Query Execute Failed: " . mysqli_stmt_error($stmt_job));
        send_json_error("Database error checking job.", 500);
    }
    mysqli_stmt_close($stmt_job);
} else {
    // Log error
    // error_log("Apply Job - Job Query Prepare Failed: " . mysqli_error($conn));
    send_json_error("Database error preparing job check.", 500);
}


// --- 3. Check if user already applied for this job ---
$already_applied = false;
$sql_check = "SELECT 1 FROM job_applications WHERE user_id = ? AND job_id = ? LIMIT 1";
$stmt_check = mysqli_prepare($conn, $sql_check);

if ($stmt_check) {
    mysqli_stmt_bind_param($stmt_check, "ii", $user_id, $job_id);
    if (mysqli_stmt_execute($stmt_check)) {
        mysqli_stmt_store_result($stmt_check);
        if (mysqli_stmt_num_rows($stmt_check) > 0) {
            $already_applied = true;
        }
    } else {
        // Log error
        // error_log("Apply Job - Check Query Execute Failed: " . mysqli_stmt_error($stmt_check));
        send_json_error("Database error checking existing application.", 500);
    }
    mysqli_stmt_close($stmt_check);
} else {
    // Log error
    // error_log("Apply Job - Check Query Prepare Failed: " . mysqli_error($conn));
    send_json_error("Database error preparing application check.", 500);
}

if ($already_applied) {
    send_json_error("You have already applied for this job.", 409); // 409 Conflict is appropriate
}


// --- 4. Insert into job_applications table ---
$application_id = null;
$sql_insert = "INSERT INTO job_applications (user_id, job_id) VALUES (?, ?)";
$stmt_insert = mysqli_prepare($conn, $sql_insert);

if ($stmt_insert) {
    mysqli_stmt_bind_param($stmt_insert, "ii", $user_id, $job_id);
    if (mysqli_stmt_execute($stmt_insert)) {
        if (mysqli_stmt_affected_rows($stmt_insert) === 1) {
            // Get the ID of the inserted application
            $application_id = mysqli_insert_id($conn);
        } else {
            // Insert failed, but no error thrown? Unlikely but possible.
            send_json_error("Failed to save application.", 500);
        }
    } else {
        // Log error (Could be duplicate entry if check failed somehow, or other DB issue)
        // error_log("Apply Job - Insert Query Execute Failed: " . mysqli_stmt_error($stmt_insert));
        send_json_error("Database error saving application.", 500);
    }
    mysqli_stmt_close($stmt_insert);
} else {
    // Log error
    // error_log("Apply Job - Insert Query Prepare Failed: " . mysqli_error($conn));
    send_json_error("Database error preparing application save.", 500);
}

// --- Close Database Connection ---
mysqli_close($conn);

// --- 5. Format Success Message and JSON Response ---
if ($application_id !== null && $job_title !== null) {
    // Format the success message with newlines
    $success_message = "Congratulations! You have applied for the job:\n";
    $success_message .= "*" . htmlspecialchars($job_title, ENT_QUOTES, 'UTF-8') . "*\n\n"; // Add emphasis if supported
    $success_message .= "Your application ID is: *" . $application_id . "*\n\n";
    $success_message .= "We will reach out to you in 7-10 days with the outcome of the application. Good Luck!\n\n";
    $success_message .= "Thank you for using EasySocial Job portal.\n";
    $success_message .= "Reply \"hi\" to restart conversation.";

    // Construct the final JSON payload
    $response_payload = [
        "data" => [
            "body_text" => $success_message
        ]
    ];

    // Output the final JSON response
    echo json_encode($response_payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

} else {
    // Should not happen if logic above is correct, but as a fallback
    send_json_error("An unexpected error occurred after applying.", 500);
}

?>