<?php
// validate_user.php

// Include the database connection file
require_once 'db_connect.php'; // Adjust path if db_connect.php is elsewhere

// Set the response header to JSON
header('Content-Type: application/json');

// Initialize the response structure
$response = [
    "data" => [
        "success" => false
    ]
];

// --- Get and Sanitize Input ---
// Check if the 'phone' parameter exists in the URL
if (!isset($_GET['phone'])) {
    // If phone parameter is missing, output the default failure response
    echo json_encode($response);
    exit(); // Stop script execution
}

// Get the raw phone number from the URL parameter
$phone_raw = trim($_GET['phone']);

// --- Prepare Phone Number for Query ---
// Normalize the input: remove leading '+' if present for comparison flexibility
$phone_normalized = ltrim($phone_raw, '+');
// Prepare the version *with* '+' (most likely stored format based on sample data)
$phone_with_plus = '+' . $phone_normalized;


// --- Prepare SQL Statement ---
// Use prepared statements to prevent SQL injection vulnerabilities
$sql = "SELECT full_name FROM users WHERE phone_number = ? OR phone_number = ?";

$stmt = mysqli_prepare($conn, $sql);

if ($stmt) {
    // Bind parameters: 'ss' means two string parameters
    // We check both the number with '+' and without, just in case of inconsistent input/storage
    mysqli_stmt_bind_param($stmt, "ss", $phone_with_plus, $phone_normalized);

    // Execute the statement
    if (mysqli_stmt_execute($stmt)) {
        // Get the result
        $result = mysqli_stmt_get_result($stmt);

        // Check if a user was found (exactly one row expected for a unique phone)
        if (mysqli_num_rows($result) > 0) {
            // Fetch the user data
            $user = mysqli_fetch_assoc($result);

            // User found, update the response to success
            $response['data']['success'] = true;
            $response['data']['lead_attributes'] = [
                'name' => $user['full_name'] ?? '' // Use null coalescing for safety
            ];
        }
        // Else: No user found, $response remains as failure (default)

    } else {
        // SQL execution failed - Log this error in a real application
        // Keep the response as generic failure for the client
        // error_log("Error executing statement: " . mysqli_stmt_error($stmt));
    }

    // Close the statement
    mysqli_stmt_close($stmt);

} else {
    // SQL preparation failed - Log this error in a real application
    // Keep the response as generic failure for the client
    // error_log("Error preparing statement: " . mysqli_error($conn));
}


// --- Close Database Connection ---
mysqli_close($conn);


// --- Output the final JSON response ---
echo json_encode($response);

?>