<?php
// job_location.php

// Include the database connection file
require_once 'db_connect.php'; // Adjust path if db_connect.php is elsewhere

// Set the response header to JSON
header('Content-Type: application/json');

// --- Define Error Response Function ---
// Helper function to send consistent error JSON and set HTTP status code
function send_error_response($message, $http_code = 404) {
    http_response_code($http_code); // Set appropriate HTTP status
    echo json_encode(["error" => ["message" => $message]]);
    // Ensure connection is closed if it exists before exiting
    if (isset($GLOBALS['conn']) && $GLOBALS['conn'] instanceof mysqli) {
       mysqli_close($GLOBALS['conn']);
    }
    exit();
}


// --- Get and Validate Job ID Input ---
$job_id = null;
if (isset($_GET['id']) && is_numeric($_GET['id'])) {
    $job_id = (int)$_GET['id'];
} else {
    // If ID is missing or invalid, send a 400 Bad Request error
    send_error_response("Job ID parameter is required and must be numeric.", 400);
}

// --- Fetch Job Location Details ---
// Select only the necessary fields for this endpoint
$sql = "SELECT
            longitude,
            latitude,
            company_name,
            location_text
        FROM jobs
        WHERE job_id = ? AND is_active = TRUE"; // Also ensure the job is active

$stmt = mysqli_prepare($conn, $sql);
$location_data = null;

if ($stmt) {
    // Bind the job_id parameter (integer)
    mysqli_stmt_bind_param($stmt, "i", $job_id);

    // Execute the statement
    if (mysqli_stmt_execute($stmt)) {
        // Get the result
        $result = mysqli_stmt_get_result($stmt);

        // Check if exactly one job was found
        if (mysqli_num_rows($result) === 1) {
            $location_data = mysqli_fetch_assoc($result);

            // --- Crucial Check: Ensure Lat/Lng exist for this job ---
            // If the main purpose is location, treat missing coords as "Not Found" for this endpoint
            if (empty($location_data['latitude']) || empty($location_data['longitude'])) {
                send_error_response("Location coordinates not available for this job.", 404);
            }

        } else {
            // Job ID not found or is inactive
            send_error_response("Job not found or is inactive.", 404);
        }

    } else {
        // SQL execution failed - Log error securely in a real app
        // error_log("Job Location Query Execution Failed: " . mysqli_stmt_error($stmt));
        send_error_response("Database query failed.", 500); // Internal Server Error
    }

    // Close the statement
    mysqli_stmt_close($stmt);

} else {
    // SQL preparation failed - Log error securely in a real app
    // error_log("Job Location Query Prepare Failed: " . mysqli_error($conn));
    send_error_response("Database query preparation failed.", 500); // Internal Server Error
}

// --- Close Database Connection ---
mysqli_close($conn);

// --- Construct the Success Response ---
// This part is only reached if the job was found, active, and had coordinates
$response_payload = [
    "data" => [
        // Explicitly cast to float for correct JSON number type
        "longitude" => (float)$location_data['longitude'],
        "latitude"  => (float)$location_data['latitude'],
        // Map database columns to desired JSON keys
        "name"      => $location_data['company_name'],   // company_name -> name
        "address"   => $location_data['location_text']   // location_text -> address
    ]
];

// --- Output the final JSON response ---
echo json_encode($response_payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

?>