<?php
// job_details.php

// Include the database connection file
require_once 'db_connect.php'; // Adjust path if db_connect.php is elsewhere

// Set the response header to JSON
header('Content-Type: application/json');

// --- Define a default structure for "Not Found" or Error ---
$notFoundResponse = [
    "data" => [
        "interactive" => [
            "type" => "button",
            "body" => ["text" => "Sorry, the requested job could not be found or is no longer available."],
            "action" => [
                "buttons" => [
                    [
                        "type" => "reply",
                        "reply" => ["id" => "back_to_jobs", "title" => "Back to Categories"]
                    ]
                ]
            ]
        ]
    ]
];

// --- Get and Validate Job ID Input ---
$job_id = null;
if (isset($_GET['id']) && is_numeric($_GET['id'])) {
    $job_id = (int)$_GET['id'];
} else {
    // If ID is missing or invalid, return the "Not Found" response
    echo json_encode($notFoundResponse);
    if (isset($conn) && $conn) {
       mysqli_close($conn);
    }
    exit();
}

// --- Fetch Job Details from Database ---
// Added posted_date to the SELECT list
$sql = "SELECT
            title,
            description,
            company_name,
            location_text,
            employment_type,
            salary_range,
            latitude,
            longitude,
            posted_date
        FROM jobs
        WHERE job_id = ? AND is_active = TRUE";

$stmt = mysqli_prepare($conn, $sql);
$job_data = null;

if ($stmt) {
    // Bind the job_id parameter (integer)
    mysqli_stmt_bind_param($stmt, "i", $job_id);

    // Execute the statement
    if (mysqli_stmt_execute($stmt)) {
        // Get the result
        $result = mysqli_stmt_get_result($stmt);

        // Check if exactly one job was found
        if (mysqli_num_rows($result) === 1) {
            $job_data = mysqli_fetch_assoc($result);
        }
        // Else: job_data remains null, will trigger the "Not Found" response later

    } else {
        // SQL execution failed - Log error
        // error_log("Job Details Query Execution Failed: " . mysqli_stmt_error($stmt));
        // Keep job_data as null
    }

    // Close the statement
    mysqli_stmt_close($stmt);

} else {
    // SQL preparation failed - Log error
    // error_log("Job Details Query Prepare Failed: " . mysqli_error($conn));
    // Keep job_data as null
}

// --- Close Database Connection (early if job not found) ---
mysqli_close($conn);

// --- Check if Job Data was Found ---
if ($job_data === null) {
    // Job not found or query failed, output the "Not Found" response
    echo json_encode($notFoundResponse);
    exit();
}

// --- Construct the Success Response ---

// Build the Body Text with specific formatting
$body_lines = [];
$body_lines[] = "*Job Description:*\n" . ($job_data['description'] ?: 'Details not provided.'); // Add emphasis with * * if supported

if (!empty($job_data['company_name'])) {
    $body_lines[] = "*Company:* " . $job_data['company_name'];
}
if (!empty($job_data['location_text'])) {
    $body_lines[] = "*Location:* " . $job_data['location_text'];
}
if (!empty($job_data['salary_range'])) {
    $body_lines[] = "*Pay:* " . $job_data['salary_range'];
}
if (!empty($job_data['employment_type'])) {
    $body_lines[] = "*Type:* " . $job_data['employment_type'];
}
$body_text = implode("\n\n", $body_lines); // Separate sections with double newline


// Build the Footer Text with formatted date
$footer_text = "Posted date unavailable"; // Default footer
if (!empty($job_data['posted_date'])) {
    $posted_timestamp = strtotime($job_data['posted_date']);
    if ($posted_timestamp) { // Check if strtotime was successful
         // Format: jS M Y (e.g., 1st Jan 2024, 2nd Feb 2024, 24th Mar 2025)
         $footer_text = "Posted on " . date('jS M Y', $posted_timestamp);
    }
}


// Create the button list with hardcoded IDs
$buttons = [];

// 1. Apply Now Button
$buttons[] = [
    "type" => "reply",
    "reply" => [
        "id" => "apply_now", // Hardcoded ID
        "title" => "Apply now"
    ]
];

// 2. View on Map Button (Conditional)
if (!empty($job_data['latitude']) && !empty($job_data['longitude'])) {
    $buttons[] = [
        "type" => "reply",
        "reply" => [
            "id" => "view_on_map", // Hardcoded ID
            "title" => "View on Map"
        ]
    ];
}

// 3. Back to Jobs Button
$buttons[] = [
    "type" => "reply",
    "reply" => [
        "id" => "back_to_jobs", // Hardcoded ID
        "title" => "Back to Jobs"
    ]
];


// Assemble the final response structure
$response = [
    "data" => [
        "interactive" => [
            "type" => "button",
            "header" => [
                "type" => "text",
                "text" => $job_data['title'] // Use job title as header
            ],
            "body" => [
                "text" => $body_text // Use the newly formatted body
            ],
            "action" => [
                "buttons" => $buttons // Use buttons with hardcoded IDs
            ],
            "footer" => [
                "text" => $footer_text // Use the formatted date footer
            ]
        ]
    ]
];

// --- Output the final JSON response ---
echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

?>