<?php
// jobs_listing.php

// Include the database connection file
require_once 'db_connect.php'; // Adjust path if db_connect.php is elsewhere

// Set the response header to JSON
header('Content-Type: application/json');

// --- Define the basic structure of the response ---
$response = [
    "data" => [
        "interactive" => [
            "type" => "list",
            "body" => [
                "text" => "Here are the available jobs in the selected category." // Default text
            ],
            "action" => [
                "button" => "View Job Details", // Default button text
                "sections" => [
                    [
                        "title" => "Job Listings", // Default section title
                        "rows" => [] // Initialize rows as an empty array
                    ]
                ]
            ]
        ]
    ]
];

// --- Get and Validate URL Parameters ---

// Category ID (Required)
$category_id = null;
if (isset($_GET['category']) && is_numeric($_GET['category'])) {
    $category_id = (int)$_GET['category'];
} else {
    // If category is missing or invalid, return empty list structure
    echo json_encode($response);
    mysqli_close($conn); // Close connection before exiting
    exit();
}

// Limit (Optional, default to 4)
$limit = 4; // Default limit
if (isset($_GET['limit']) && is_numeric($_GET['limit']) && (int)$_GET['limit'] > 0) {
    $limit = (int)$_GET['limit'];
}

// Page (Optional, default to 1)
$current_page = 1;
if (isset($_GET['page'])) {
    // Extract number from "Page_X" format
    if (preg_match('/^Page_(\d+)$/', $_GET['page'], $matches)) {
        $page_num = (int)$matches[1];
        if ($page_num > 0) {
            $current_page = $page_num;
        }
    }
    // If format is invalid or number is not positive, it defaults to 1
}

// Calculate Offset for SQL Query
$offset = ($current_page - 1) * $limit;

// --- Fetch Total Count for Pagination ---
$total_jobs = 0;
$sql_count = "SELECT COUNT(*) as total FROM jobs WHERE category_id = ? AND is_active = TRUE";

$stmt_count = mysqli_prepare($conn, $sql_count);
if ($stmt_count) {
    mysqli_stmt_bind_param($stmt_count, "i", $category_id);
    if (mysqli_stmt_execute($stmt_count)) {
        $result_count = mysqli_stmt_get_result($stmt_count);
        $count_data = mysqli_fetch_assoc($result_count);
        $total_jobs = (int)$count_data['total'];
    } else {
        // Log error
        // error_log("Count Query Execution Failed: " . mysqli_stmt_error($stmt_count));
    }
    mysqli_stmt_close($stmt_count);
} else {
    // Log error
    // error_log("Count Query Prepare Failed: " . mysqli_error($conn));
}

// Calculate total pages
$total_pages = ($total_jobs > 0) ? ceil($total_jobs / $limit) : 0;


// --- Fetch Job Data for the Current Page ---
$jobs_found_on_page = false;
// We only query for jobs if the requested category actually has jobs
if ($total_jobs > 0 && $current_page <= $total_pages) {
    // Select necessary fields, including the ones needed for description
    $sql_jobs = "SELECT job_id, title, company_name, location_text, job_category
                 FROM jobs
                 WHERE category_id = ? AND is_active = TRUE
                 ORDER BY posted_date DESC -- Or another relevant order
                 LIMIT ? OFFSET ?";

    $stmt_jobs = mysqli_prepare($conn, $sql_jobs);

    if ($stmt_jobs) {
        // Bind parameters: category_id (integer), limit (integer), offset (integer)
        mysqli_stmt_bind_param($stmt_jobs, "iii", $category_id, $limit, $offset);

        if (mysqli_stmt_execute($stmt_jobs)) {
            $result_jobs = mysqli_stmt_get_result($stmt_jobs);

            if (mysqli_num_rows($result_jobs) > 0) {
                $jobs_found_on_page = true;
                // Customize response text and title based on found category name
                $first_job = mysqli_fetch_assoc($result_jobs); // Fetch first to get category name
                if (!empty($first_job['job_category'])) {
                     $response['data']['interactive']['body']['text'] = "Please choose a job from the " . $first_job['job_category'] . " category.";
                     $response['data']['interactive']['action']['sections'][0]['title'] = $first_job['job_category'] . " Jobs";
                }
                mysqli_data_seek($result_jobs, 0); // Reset pointer to beginning

                // Loop through results and format them
                while ($job = mysqli_fetch_assoc($result_jobs)) {
                    // Format description string
                    $description_parts = [];
                    if (!empty($job['company_name'])) {
                        $description_parts[] = $job['company_name'];
                    }
                    if (!empty($job['location_text'])) {
                        $description_parts[] = $job['location_text'];
                    }
                    $job_description = implode(', ', $description_parts);
                    if (empty($job_description)) {
                       $job_description = "-"; // Placeholder if no company/location
                    }


                    // Create the row structure
                    $row_item = [
                        "id" => (string)$job['job_id'], // Job ID for selection
                        "title" => $job['title'],
                        "description" => $job_description
                    ];
                    // Add the formatted row
                    $response['data']['interactive']['action']['sections'][0]['rows'][] = $row_item;
                }
            }
            // Else: No jobs found on this specific page (might happen if page > total_pages requested, handled above)

        } else {
             // Log error
            // error_log("Jobs Query Execution Failed: " . mysqli_stmt_error($stmt_jobs));
        }
        mysqli_stmt_close($stmt_jobs);
    } else {
        // Log error
        // error_log("Jobs Query Prepare Failed: " . mysqli_error($conn));
    }
} else {
     // If no jobs found at all for the category, update the body text
     if($total_jobs === 0) {
        $response['data']['interactive']['body']['text'] = "Sorry, no active jobs found in this category currently.";
     }
     // If page requested is beyond total pages, the loop won't run, rows remain empty.
}


// --- Add Pagination Options ---

// "See more jobs" if current page is not the last page
if ($current_page < $total_pages) {
    $response['data']['interactive']['action']['sections'][0]['rows'][] = [
        "id" => "Page_" . ($current_page + 1), // ID to trigger loading the next page
        "title" => "➡️ See more jobs",
        "description" => "View the next set of results"
    ];
}
// "Back to Jobs" if this is the last page AND there were jobs in the category
else if ($current_page == $total_pages && $total_jobs > 0) {
     $response['data']['interactive']['action']['sections'][0]['rows'][] = [
        "id" => "back_to_jobs", // Static ID to trigger going back
        "title" => "⏪ Back to Categories",
        "description" => "View job categories again"
    ];
}


// --- Close Database Connection ---
mysqli_close($conn);

// --- Output the final JSON response ---
echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

?>